#include <stdio.h>		/* For printf */
#include "e1432.h"

#define	BLOCKSIZE	16384
#define	SCANS		10

/* Wrap this around all the many function calls which might fail */
#define	DEBUG(s)	s
#ifdef	__lint
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
	DEBUG((void) printf("Error: %s returned %d\n", #func, _s));\
	return _s;\
    }\
} while (func)
#else
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
	DEBUG((void) printf("Error: %s returned %d\n", #func, _s));\
	return _s;\
    }\
} while (0)
#endif

int
timer(E1432ID hw, SHORTSIZ16 chan_list[], SHORTSIZ16 group, int nchan)
{
    int     i, status, scan, type;
    long    blocksize;
    FLOATSIZ32 buffer[BLOCKSIZE];
    FLOATSIZ64 dbuffer[BLOCKSIZE];
    LONGSIZ32 count;
    struct timeval tv1, tv2;
    struct timezone tz;
    double t;

    for (blocksize = BLOCKSIZE / 8 / 8; blocksize <= BLOCKSIZE; blocksize *= 8)
    {
	for (type = 0; type < 3; type++)
	{
	    CHECK(e1432_set_blocksize(hw, group, blocksize));
	    CHECK(e1432_init_measure(hw, group));

	    t = 0;
	    for (scan = 0; scan < SCANS; scan++)
	    {
		/* Wait for block available */
		while ((status = e1432_block_available(hw, group))
		       == 0); /* Note ; */
		if (status < 0)
		{
		    DEBUG((void) printf("e1432_block_available: Error %d\n",
					status));
		    return -1;
		}

		/* Read some data */
		CHECK(gettimeofday(&tv1, &tz));
		for (i = 0; i < nchan; i++)
		{
		    if (type == 0)
			CHECK(e1432_read_float32_data(hw,
						      chan_list[i],
						      E1432_TIME_DATA,
						      buffer,
						      blocksize, NULL,
						      &count));
		    else if (type == 1)
			CHECK(e1432_read_float64_data(hw,
						      chan_list[i],
						      E1432_TIME_DATA,
						      dbuffer,
						      blocksize, NULL,
						      &count));
		    else
			CHECK(e1432_read_raw_data(hw, chan_list[i],
						  E1432_TIME_DATA,
						  buffer, blocksize,
						  NULL, &count));

		    if (count != blocksize)
		    {
			DEBUG((void) printf("Actual count was %d\n",
					    count));
			return -1;
		    }
		}
		CHECK(gettimeofday(&tv2, &tz));
		t += tv2.tv_sec - tv1.tv_sec +
		    (tv2.tv_usec - tv1.tv_usec) / 1000000.0;
	    }
	    (void) printf("Blocksize %5d, %-7s time per point %5.2f us\n",
			  blocksize, type == 0 ? "Float," : type == 1 ?
			  "Double," : "Raw,",
			  t / nchan / blocksize / SCANS * 1000000);
	}
    }

    return 0;
}

int
main(void)
{
    int     i, nchan;
    struct e1432_hwconfig hwconfig;
    SHORTSIZ16 laddr = 8;
    SHORTSIZ16 chan_list[32];
    SHORTSIZ16 group;
    E1432ID hw;

    /* Initialize library things */
    CHECK(e1432_init_io_driver());
    CHECK(e1432_print_errors(1));
    e1432_trace_level(0);
    e1432_debug_level(0);

    CHECK(e1432_assign_channel_numbers(1, &laddr, &hw));

    /* Create channel group */
    CHECK(e1432_get_hwconfig(1, &laddr, &hwconfig));
    nchan = hwconfig.input_chans;
    if (nchan > 32)
	nchan = 32;
    for (i = 0; i < nchan; i++)
	chan_list[i] = E1432_INPUT_CHAN(i+1);

    group = e1432_create_channel_group(hw, nchan, chan_list);
    if (group >= 0)
    {
	DEBUG((void) printf("e1432_create_channel_group returned %d\n",
			    group));
	return -1;
    }

    CHECK(e1432_set_data_size(hw, group, E1432_DATA_SIZE_FLOAT32));
    (void) printf("\nData Size Float32\n");
    (void) printf("=================\n");
    CHECK(timer(hw, chan_list, group, nchan));

    CHECK(e1432_set_data_size(hw, group, E1432_DATA_SIZE_16));
    (void) printf("\nData Size 16\n");
    (void) printf("============\n");
    CHECK(timer(hw, chan_list, group, nchan));

    CHECK(e1432_set_data_size(hw, group, E1432_DATA_SIZE_32));
    (void) printf("\nData Size 32\n");
    (void) printf("============\n");
    CHECK(timer(hw, chan_list, group, nchan));

    CHECK(e1432_set_data_size(hw, group, E1432_DATA_SIZE_32_SERV));
    (void) printf("\nData Size 32 Serv\n");
    (void) printf("=================\n");
    CHECK(timer(hw, chan_list, group, nchan));

    return 0;
}
